package civvi.osgi.desktop.tail.wizard;

import java.awt.BorderLayout;
import java.io.File;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;

import org.jdesktop.application.Action;
import org.jdesktop.application.Application;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.Task;
import org.jdesktop.swingx.JXPanel;

import civvi.osgi.desktop.swingx.wizard.AbstractHeaderStepPanel;
import civvi.osgi.desktop.swingx.wizard.WizardModel;
import civvi.osgi.desktop.tail.TailEditable;

/**
 * A panel for creating a new {@link TailEditable}.
 * 
 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
 * @since 14 Apr 2009
 */
public class TailEditableStepPanel extends AbstractHeaderStepPanel {
	private static final long serialVersionUID = 1719065355722609971L;
	
	private final JTextField filePathField;
	private final JFileChooser fileChooser;
	private final JLabel infoLabel;
	
	private WizardModel<?> wizardModel;
	
	/**
	 * Default constructor.
	 */
	public TailEditableStepPanel() {
		this.header.setTitle(getResourceMap().getString("title"));
		this.header.setDescription(getResourceMap().getString("description"));
		this.header.setIcon(getResourceMap().getIcon("icon.large"));
		
		final JXPanel panel = new JXPanel(new BorderLayout());
		panel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		final JXPanel northPanel = new JXPanel(new BorderLayout(3, 3));
		final JLabel label = new JLabel(getResourceMap().getString("label.file"));
		northPanel.add(label , BorderLayout.WEST);
		northPanel.add(this.filePathField = new JTextField(40), BorderLayout.CENTER);
		this.filePathField.getDocument().addDocumentListener(new DocumentHandler());
		label.setLabelFor(this.filePathField);
		this.filePathField.requestFocusInWindow();
		northPanel.add(new JButton(getContext().getActionMap(this).get("selectFile")), BorderLayout.EAST);
		panel.add(northPanel, BorderLayout.NORTH);
		panel.add(this.infoLabel = new JLabel(), BorderLayout.SOUTH);
		
		this.fileChooser = new JFileChooser();
//		this.fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
//		this.fileChooser.addActionListener(new ActionHandler());
		setContent(panel);

		updateStatus(false, null);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void associate(WizardModel<?> model) {
		this.wizardModel = model;
		
		final File selected = (File) wizardModel.getValue("selected.file");
		this.filePathField.setText(selected != null ? selected.getAbsolutePath() : null);
	}
	
	/**
	 * @return the application context.
	 */
	private ApplicationContext getContext() {
		return Application.getContext();
	}
	
	/**
	 * @return the resource map for the view.
	 */
	public ResourceMap getResourceMap() {
		return getContext().getResourceMap(getClass());
	}

	/**
	 * Performs selection of a file.
	 */
	@Action
	public void selectFile() {
		final File file = new File(this.filePathField.getText());

		// attempt to file or directory described in the field 
		if (file.exists()) {
			this.fileChooser.setSelectedFile(file);
		}

		if (this.fileChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
			final File selected = this.fileChooser.getSelectedFile();
//			this.wizardModel.putValue("selected.file", selected);
			this.filePathField.setText(selected.getAbsolutePath());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean validates() {
		return this.wizardModel.getValue("selected.file") != null;
	}

	/**
	 * Checks the file for it's existence.
	 *
	 * @param file the file to check.
	 * @return the task to farm off into the background.
	 */
	private Task<Boolean, Object> checkFile(File file) {
		this.wizardModel.putValue("selected.file", null);
		return new CheckFileTask(getContext().getApplication(), file); 
	}

	/**
	 * Updates the status label.
	 *
	 * @param exists if {@code true} the user will be told the file exists.
	 * @param file the file it attempted to to find.
	 */
	private void updateStatus(boolean exists, File file) {
		final ResourceMap resourceMap = getResourceMap();
		
		if (exists) {
			infoLabel.setIcon(resourceMap.getIcon("file.found.icon"));
			infoLabel.setText(resourceMap.getString("file.found.text"));
			wizardModel.putValue("selected.file", file);
		} else {
			infoLabel.setIcon(resourceMap.getIcon("file.notFound.icon"));
			infoLabel.setText(resourceMap.getString("file.notFound.text"));
		}
	}


	// --- Inner Classes ---

	/**
	 * Checks the existance of the file in the background.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 14 Apr 2009
	 */
	private class CheckFileTask extends Task<Boolean, Object> {
		private final File file;

		public CheckFileTask(Application<?> application, File file) {
			super(application);
			this.file = file;
		}

		@Override
		protected Boolean doInBackground() throws Exception {
			return this.file.exists() && this.file.isFile();
		}
		
		@Override
		protected void succeeded(Boolean result) {
			updateStatus(result, this.file);
		}
	}

	/**
	 * Handles changes to the document.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 14 Apr 2009
	 */
	private class DocumentHandler implements DocumentListener {

		@Override
		public void changedUpdate(DocumentEvent e) {
			try {
				final String string = e.getDocument().getText(0, e.getDocument().getLength());
				getContext().getTaskService().execute(checkFile(new File(string)));
			} catch (BadLocationException ble) {
				throw new RuntimeException(ble); // should never happen!
			}
		}

		@Override
		public void insertUpdate(DocumentEvent e) {
			changedUpdate(e);
		}

		@Override
		public void removeUpdate(DocumentEvent e) {
			changedUpdate(e);
		}
	}
}
