package demo.poll;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.psddev.dari.db.Grouping;
import com.psddev.dari.db.Query;
import com.psddev.dari.db.Record;

public class Poll extends Record {

    @Indexed private String question;

    // --- getters / setters

    public String getQuestion() {
        return question;
    }
    public void setQuestion(String question) {
        this.question = question;
    }

    // --- helper methods ---

    /** Creates a new Poll object along with its corresponding answers and
     *  saves them. */
    public static Poll createPoll(String question, List<String> answerStrings) {

        Poll poll = new Poll();
        poll.setQuestion(question);
        poll.save();

        if (answerStrings != null) for (String answerString : answerStrings) {
            Answer answer = new Answer();
            answer.setPoll(poll);
            answer.setAnswer(answerString);
            answer.save();
        }

        return poll;
    }

    /** Helper method that queries for the answers to the current poll. */
    public List<Answer> getAnswers() {
        return Query.from(Answer.class).where("poll = ?", this).selectAll();
    }

    /** Submits a user's answer to the poll and returns a saved response. */
    public UserResponse submit(User user, Answer answer) {

        if (user != null && answer != null && this.equals(answer.getPoll())) {
            UserResponse response = new UserResponse();
            response.setUser(user);
            response.setAnswer(answer);
            response.save();
            return response;

        } else {
            return null;
        }
    }

    /** Gets the poll results as a mapping of answer to the number of responses
     *  with that answer. */
    public Map<String, Long> getPollResults() {

        Map<String, Long> pollResults = new HashMap<String, Long>();

        Query<UserResponse> responseQuery = Query.from(UserResponse.class)
                .where("answer/poll = ?", this);

        for (Grouping<UserResponse> grouping : responseQuery.groupBy("answer")) {

            List<Object> keys = grouping.getKeys();
            Object key0 = keys.get(0);

            if (key0 instanceof Answer) {
                Answer answer = (Answer) key0;
                long answerCount = grouping.getCount();

                pollResults.put(answer.getAnswer(), answerCount);
            }
        }

        return pollResults;
    }

    /** Returns the total number of responses to this poll. */
    public long getPollResponseCount() {
        return Query.from(UserResponse.class)
                .where("answer/poll = ?", this)
                .count();
    }
}